<?php
require_once '../includes/header.php';
require_once '../includes/functions.php';

// Check if user has permission to view logs
// Check if user has any logs-related permissions
$logsPermissions = [
    'view_logs', 'view_system_logs', 'manage_logs'
];

if (!hasAnyPermission($logsPermissions)) {
    $_SESSION['error'] = "You don't have permission to view logs.";
    header('Location: dashboard.php');
    exit;
}

if (false && !hasPermission('view_logs')) { // Disable the old check
    $_SESSION['error'] = "You don't have permission to view system logs.";
    header('Location: ../pages/dashboard.php');
    exit();
}

try {
    // Get search parameters
    $search = isset($_GET['search']) ? trim($_GET['search']) : '';
    $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $per_page = 20;
    $offset = ($page - 1) * $per_page;

    // Build search condition
    $search_condition = '';
    $params = [];
    if (!empty($search)) {
        $search_condition = "WHERE l.description LIKE :search 
                            OR u.username LIKE :search 
                            OR l.action_type LIKE :search";
        $params[':search'] = "%$search%";
    }

    // Get total records for pagination
    $total_query = "SELECT COUNT(*) as total 
                    FROM system_logs l 
                    LEFT JOIN users u ON l.user_id = u.id 
                    $search_condition";
    $stmt = $pdo->prepare($total_query);
    $stmt->execute($params);
    $total_records = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    $total_pages = ceil($total_records / $per_page);

    // Get logs with pagination
    $query = "SELECT l.*, u.username 
              FROM system_logs l 
              LEFT JOIN users u ON l.user_id = u.id 
              $search_condition 
              ORDER BY l.created_at DESC 
              LIMIT :offset, :per_page";
    
    $stmt = $pdo->prepare($query);
    $params[':offset'] = $offset;
    $params[':per_page'] = $per_page;
    $stmt->execute($params);
    $logs = $stmt->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    error_log("Error in logs.php: " . $e->getMessage());
    $_SESSION['error'] = "An error occurred while fetching logs.";
    $logs = [];
    $total_pages = 0;
}
?>

<!-- Content Wrapper. Contains page content -->
<div class="content-wrapper">
    <!-- Content Header (Page header) -->
    <div class="content-header">
        <div class="container-fluid">
            <div class="row mb-2">
                <div class="col-sm-6">
                    <h1 class="m-0">System Logs</h1>
                </div>
                <div class="col-sm-6">
                    <ol class="breadcrumb float-sm-right">
                        <li class="breadcrumb-item"><a href="../pages/dashboard.php">Home</a></li>
                        <li class="breadcrumb-item active">System Logs</li>
                    </ol>
                </div>
            </div>
        </div>
    </div>

    <!-- Main content -->
    <section class="content">
        <div class="container-fluid">
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">System Activity Logs</h3>
                    <div class="card-tools">
                        <form action="" method="GET" class="form-inline">
                            <div class="input-group">
                                <input type="text" name="search" class="form-control" placeholder="Search logs..." value="<?php echo htmlspecialchars($search); ?>">
                                <div class="input-group-append">
                                    <button type="submit" class="btn btn-default">
                                        <i class="fas fa-search"></i>
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
                <div class="card-body table-responsive p-0">
                    <table class="table table-hover text-nowrap">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Action</th>
                                <th>Description</th>
                                <th>User</th>
                                <th>IP Address</th>
                                <th>Timestamp</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (!empty($logs)): ?>
                                <?php foreach ($logs as $log): ?>
                                    <tr>
                                        <td><?php echo $log['id']; ?></td>
                                        <td>
                                            <span class="badge <?php echo getActionBadgeClass($log['action_type']); ?>">
                                                <?php echo getActionDescription($log['action_type']); ?>
                                            </span>
                                        </td>
                                        <td><?php echo htmlspecialchars($log['description']); ?></td>
                                        <td><?php echo htmlspecialchars($log['username'] ?? 'System'); ?></td>
                                        <td><?php echo htmlspecialchars($log['ip_address']); ?></td>
                                        <td><?php echo formatTimestamp($log['created_at']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <tr>
                                    <td colspan="6" class="text-center">No logs found</td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
                <?php if ($total_pages > 1): ?>
                    <div class="card-footer clearfix">
                        <ul class="pagination pagination-sm m-0 float-right">
                            <?php if ($page > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?page=<?php echo ($page - 1); ?><?php echo $search ? '&search=' . urlencode($search) : ''; ?>">Previous</a>
                                </li>
                            <?php endif; ?>
                            
                            <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                                <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $i; ?><?php echo $search ? '&search=' . urlencode($search) : ''; ?>"><?php echo $i; ?></a>
                                </li>
                            <?php endfor; ?>
                            
                            <?php if ($page < $total_pages): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?page=<?php echo ($page + 1); ?><?php echo $search ? '&search=' . urlencode($search) : ''; ?>">Next</a>
                                </li>
                            <?php endif; ?>
                        </ul>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </section>
</div>

<?php require_once '../includes/footer.php'; ?> 